﻿"""Implementation of a transport for flying
"""

import vizmat

import transportation


class Flying(transportation.AccelerationTransport):
	"""Implementation of a flying transport. Forward and backward can change
	elevation.
	"""
	
	def __init__(self,
					height=0,
					acceleration=4.0, # in meters per second per second, lower accelerations can be obtained by using a smaller mag on the input, e.g. pressing the joystick lower
					maxSpeed=10.44, # in meters per second, as a reference 1.4m/s is a typical walking speed, 10.44 is a very fast run
					rotationAcceleration=90.0, # in degrees per second per second
					maxRotationSpeed=120.0, # in degrees per second
					autoBreakingDragCoef=0.1, # determines how quickly the walking transport will stop 
					dragCoef=0.0001,
					rotationAutoBreakingDragCoef=0.2, # determines how quickly the walking transport will stop 
					rotationDragCoef=0.0001, # normal drag coef
					**kwargs):
		
		super(Flying, self).__init__(acceleration=acceleration,
								maxSpeed=maxSpeed,
								rotationAcceleration=rotationAcceleration,
								maxRotationSpeed=maxRotationSpeed,
								autoBreakingDragCoef=autoBreakingDragCoef,
								dragCoef=dragCoef,
								rotationAutoBreakingDragCoef=rotationAutoBreakingDragCoef,
								rotationDragCoef=rotationDragCoef,
								**kwargs)
		self.setPosition(0, height, 0)
		self._moveOnHorizontal = False
	
	def moveForward(self, mag=1):
		"""Moves the transport forward."""
		self._Ap[2] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def moveBackward(self, mag=1):
		"""Moves the transport backward."""
		self._Ap[2] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def moveLeft(self, mag=1):
		"""Moves the transport left."""
		self._Ap[0] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def moveRight(self, mag=1):
		"""Moves the transport right."""
		self._Ap[0] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def moveUp(self, mag=1):
		"""Moves the transport up."""
		self._Ap[1] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def moveDown(self, mag=1):
		"""Moves the transport down."""
		self._Ap[1] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def lookDown(self, mag=1):
		"""Pitches the transport down."""
		self._Ar[1] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def lookUp(self, mag=1):
		"""Pitches the transport up."""
		self._Ar[1] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def turnLeft(self, mag=1):
		"""Moves the transport down"""
		self._Ar[0] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def turnRight(self, mag=1):
		"""Moves the transport down"""
		self._Ar[0] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def getAccelerationRotationMatrix(self):
		"""Returns the acceleration rotation matrix, i.e. the matrix used to
		determine reference frame for acceleration.
		
		@return vizmat.Transform()
		"""
		rotMat = vizmat.Transform()
		rotMat.setEuler([self.getEuler()[0], self.getEuler()[1], 0])
		return rotMat


